import { string } from 'alga-js'

/**
 * Get payment details for a fulfillment customer invoice
 * Uses service token to fetch bank details from the default organization
 */
export default defineEventHandler(async (event) => {
  const config = useRuntimeConfig()
  const serviceToken = config.api.idempieretoken
  const invoiceId = getRouterParam(event, 'id')

  if (!serviceToken || !invoiceId) {
    return { status: 400, message: 'Missing required parameters' }
  }

  try {
    // Get DEFAULT_ORG_ID from env
    const defaultOrgId = process.env.DEFAULT_ORG_ID || '1000000'

    // Get invoice details
    const invoiceUrl = `${config.api.url}/models/c_invoice/${invoiceId}?$select=DocumentNo,GrandTotal,C_Currency_ID,DateInvoiced,C_PaymentTerm_ID&$expand=C_PaymentTerm_ID($select=NetDays)`
    const invoiceRes: any = await $fetch(invoiceUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        Accept: 'application/json',
        Authorization: 'Bearer ' + serviceToken
      }
    })

    const documentNo = invoiceRes?.DocumentNo || invoiceId
    const grandTotal = invoiceRes?.GrandTotal || 0
    const currency = invoiceRes?.C_Currency_ID?.identifier || 'EUR'
    const dateInvoiced = invoiceRes?.DateInvoiced
    const netDays = invoiceRes?.C_PaymentTerm_ID?.NetDays || 0

    // Calculate due date
    let dueDate = ''
    if (dateInvoiced) {
      const invoiceDateObj = new Date(dateInvoiced)
      invoiceDateObj.setDate(invoiceDateObj.getDate() + netDays)
      dueDate = invoiceDateObj.toISOString()
    }

    // Get default bank account from default organization
    const bankUrl = `${config.api.url}/models/c_bankaccount?$filter=${string.urlEncode(`IsDefault eq true and AD_Org_ID eq ${defaultOrgId}`)}`
    const bankRes: any = await $fetch(bankUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        Accept: 'application/json',
        Authorization: 'Bearer ' + serviceToken
      }
    })
    const bankAccount = bankRes?.records?.[0]

    // Get organization/company name
    const orgUrl = `${config.api.url}/models/ad_org?$filter=${string.urlEncode(`AD_Org_ID eq ${defaultOrgId}`)}`
    const orgRes: any = await $fetch(orgUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        Accept: 'application/json',
        Authorization: 'Bearer ' + serviceToken
      }
    })
    const companyName = orgRes?.records?.[0]?.companyname || orgRes?.records?.[0]?.Name || 'Company'

    return {
      status: 200,
      data: {
        invoice: {
          documentNo,
          grandTotal,
          currency,
          dateInvoiced,
          dueDate
        },
        bank: {
          companyName,
          iban: bankAccount?.IBAN || '',
          bic: bankAccount?.BBAN || '', // BBAN field contains BIC
          bankName: bankAccount?.Name || bankAccount?.C_Bank_ID?.identifier || ''
        }
      }
    }
  } catch (err: any) {
    console.error('Error fetching payment details:', err)
    return { status: 500, message: 'Failed to fetch payment details' }
  }
})
